"""Transcode a media using a given template."""

# Import built-in modules
import json
import os
import sys

# Import third-party modules
import nuke

# In order to be able to import local modules from smartElements we must add
# the smartElement's packages root to our sys path. Only then are we able to
# import local modules.
PACKAGE_ROOT = os.path.abspath(os.path.join(__file__, "..", "..", ".."))
sys.path.append(PACKAGE_ROOT)

# Import local modules
from smartElements import paths
from smartElements.media import Media
from smartElements.constants import VIDEO_EXTENSIONS


def main(source, element_dest_root, template):
    """Transcode a media using a given template.

    Args:
        source (str): Absolute path of source media to render preview for.
        element_dest_root (str): Absolute path of the element root directory
            to write transcoding to.
        template (str): Absolute path of the template to use.

    Raises:
        RuntimeError: When the template does not contain the required
            minimum node setup. It needs a Read node called 'SOURCE' and
            a Write node called 'TARGET'.

    """
    # Make sure we sanitize all paths as Nuke does not understand backward
    # slashes.
    source = source.replace("\\", "/")

    if not os.path.isfile(template):
        raise RuntimeError(
            "No such template path found under the path: {0}".format(template)
        )

    nuke.scriptOpen(template)

    read = nuke.toNode("SOURCE")
    if not read:
        raise RuntimeError(
            "No such Read node named 'SOURCE' found. Make sure a Read node "
            "with that node name is contained. This is the source to "
            "transcode."
        )

    # Set up the path and frame range values to render. We must distinguish
    # between an image sequence and a provided video file. If we are handling
    # an image sequence then use the provided ``start``  and ``end`` value.
    # However, when handling a video file, we cannot use these values as they
    # don't reflect the correct frame range. We must then use the video's
    # native frame range which we get by using read["file"].fromUserText.
    extension = os.path.splitext(source)[1]
    if extension in VIDEO_EXTENSIONS:
        # Using setValue will not work as Nuke will not detect the source's
        # frame range. We need fromUserText so that Nuke will set up the
        # frame range correctly. For additional information see:
        # https://community.foundry.com/discuss/post/891640
        # Using fromUserText will both set the file knob value and the frame
        # range.
        read["file"].fromUserText(source)
    else:
        read["file"].setValue(source)
        media = Media(source)
        start = media.first
        end = media.last
        for name in ("first", "origfirst"):
            read[name].setValue(start)
        for name in ("last", "origlast"):
            read[name].setValue(end)

    start = read.frameRange().first()
    end = read.frameRange().last()

    write = nuke.toNode("TARGET")
    if not write:
        raise RuntimeError(
            "No such Write node named 'TARGET' found. Make sure a Write node "
            "with that node name is contained. This is the Write node that "
            "renders the transcode source."
        )

    dest = _assemble_dest(
        element_dest_root,
        source,
        write["file_type"].value()
    )
    write["file"].setValue(dest)

    # Save .nk file that created this preview in element's meta directory.
    script_path = os.path.join(
        os.path.dirname(dest),
        ".meta",
        "transcode.nk"
    )
    nuke.scriptSave(script_path)

    # Add after frame render callbacks to keep track of render progress.
    nuke.addAfterFrameRender(increase_progress)

    # Execute the seq rendering.
    nuke.execute(write, start, end)


def _assemble_dest(element_root, source, extension):
    """Assemble the destination to render to.

    Returns:
        str: Absolute path of sequence to render to.

    """
    media = Media(source)
    file_name = "{0}.%04d.{1}".format(media.base, extension)

    return os.path.join(element_root, file_name).replace("\\", "/")


def increase_progress():
    """Increase the render progress counter by 1."""
    if not os.path.isfile(progress_path):
        return
    
    with open(progress_path, "r") as file_:
        content = json.load(file_)
        current_new = content["current"] + 1
        content["current"] = current_new
    with open(progress_path, "w") as file_:
        json.dump(content, file_)


if __name__ == "__main__":
    source = sys.argv[1]
    element_dest_root = sys.argv[2]
    template = sys.argv[3].replace("template=", "")
    progress_path = paths.get_element_progress_path(element_dest_root)
    paths.ensure_directory(os.path.dirname(progress_path))

    main(source, element_dest_root, template)
