"""Send selected media to DJV.

The media of the selected elements will be send to the DJV media player. In
order to make that happen, you will of course need to tell smartElements where
the DJV executable is located. This gets done by setting the environment
variable CRAGL_SMARTELEMENTS_DJV_EXECUTABLE to the executable path of DJV.

"""

# Import built-in modules
import os
import subprocess

# Import local modules
from smartElements.media_commands.base_media_command import BaseMediaCommand


class SendToDJV(BaseMediaCommand):
    """Send selected media to DJV."""

    EXE_PATH = "CRAGL_SMARTELEMENTS_DJV_EXECUTABLE"

    def get_executable(self):
        """Get the DJV executable.

        Returns:
            str: Absolute path to the DJV executable.

        Raises:
            EnvironmentError: When the environment variable
                CRAGL_SMARTELEMENTS_DJV_EXECUTABLE which points to the DJV
                executable is not set.
            ValueError: When the executable does not point to a file.

        """
        exe_path = os.environ.get(self.EXE_PATH)
        if not exe_path:
            raise EnvironmentError(
                "Environment variable '{}' which should point to the DJV "
                "executable is not set. Please make sure to set it."
                "".format(self.EXE_PATH)
            )

        if not os.path.isfile(exe_path):
            raise ValueError(
                "Environment variable '{}' does not point to a file. Make "
                "sure it points to the DJV executable.".format(self.EXE_PATH)
            )

        return exe_path

    def assemble_command(self, media):
        """Assemble the command to run in order to send media to DJV.

        media (:obj:`list` of :obj:`smartElements.media.Media`):
            Sequence of selected Media instances to use for command
            assembly.

        Returns:
            :obj:`list` of :obj:`str`: The command to run in order to send
                media to DJV.

        """
        args = [
            '"{}"'.format(self.get_executable())
        ]
        for media_ in media:
            args.append('"{}"'.format(media_.path))

        return args

    def label(self):
        """Return the label to use in the UI for this command.

        Returns:
            str: The label to use for the command in the UI.

        """
        return "Send to DJV"

    def process(self, media):
        """Send selected media to DJV.

        Args:
            media (:obj:`list` of :obj:`smartElements.media.Media`):
                Sequence of selected Media instances. Will send all selected
                media to DJV.

        """
        command = " ".join(self.assemble_command(media))
        self.logger.info("Running %s using command: %s", self.name, command)

        subprocess.Popen(command, shell=True)
