"""Base class for a media command to trigger action for selected media.

This abstract base class provides base functionality for all media command
classes. You will need to implement the following abstract methods:

    label (None -> str): Return the label to show in the UI for this command.
    process (media -> None): Process action for the selected media. This method
        dictates what to do when clicking the media command or executing it
        manually. You can raise any built-in Error class in here and these
        will be shown as dialog to the user.

When executing the process method from the smartElements GUI then the `media`
arguments will be set automatically based on the selected elements. When
triggering the method manually you can create Media instance manually by
specifying a list of `smartElements.media.Media` instances. This is how you
can create a custom command and trigger it from the terminal. The following
uses a command class called CustomCC that inherits from BaseMediaCommand and
implements `label` and `process`. Keep in mind that this class does not
exist and is only used for demonstration purposes.:

    # Import the needed modules
    >>> from smartElements.media import Media
    >>> from smartElements.media_commands.custom import CustomCC

    # Let's create a custom command instance.
    >>> custom = customCC()

    # This is our media to process. If a media is a file sequence simply point
    # to any file of it. The Media instance will discover all related files
    # of the sequence.
    >>> my_media = [
    ...     Media("path/to/footage/a.1001.exr"),
    ...     Media("path/to/footage/b.1001.exr"),
    ...     Media("path/to/footage/c.1001.exr")
    ... ]

    # Process the media.
    >>> customCC.process(my_media)

For more media commands examples have a look at the other media commands in
smartElements.media_commands.

NOTE:
    You need to register the media command to smartElements so that it shows up
    in smartElements. Navigate to smartElements.media_commands.__init__.py,
    import your class and call
    `osl.register_media_command(<media command class>)`. Have a look at this
    module for more information.

"""

# Import built-in modules
import abc
import logging

# Import local modules
from smartElements.clogging import LEVEL


class BaseMediaCommand(object):
    """Base class for a media command to trigger action for selected media."""

    __metaclass__ = abc.ABCMeta

    def __init__(self):
        """Initialize the BaseMediaCommand instance."""
        self.logger = logging.getLogger(self.__class__.__name__)
        self.logger.setLevel(LEVEL)

    @property
    def name(self):
        """Return the name of the media command.

        Returns:
            str: The name of the media command.

        """
        return self.__class__.__name__

    @abc.abstractmethod
    def label(self):
        """Return the label to show in the UI for this command.

        Each command has a label. Here we return the text to use for the
        command.

        Returns:
            str: The label to use for the command in the UI.

        """
        return ""

    @abc.abstractmethod
    def process(self, media):
        """Process action for the selected media.

        Every concrete implementation must implement this function. It dictates
        what to do when triggering the command.

        Args:
            media (:obj:`list` of :obj:`smartElements.media.Media`): Sequence
                of selected Media instances to process. A MediaInstance holds
                information about the underlying media and its frame range.

        """
        pass
