"""Abstract base class of a processor."""

# Import built-in modules
import abc
import json
import logging
import os
import time
from threading import Lock

# Import local modules
from smartElements.clogging import LEVEL

# The global lock to use shared resources among all processors.
LOCK = Lock()


class BaseIngestProcessor(object):
    """Abstract base class of a processor that runs when ingesting media."""

    __metaclass__ = abc.ABCMeta

    # The default settings to apply for a processor if not otherwise specified
    # in the smartElement's processor settings.
    DEFAULT_SETTINGS = {
        "enable": True,
    }

    def __init__(self, job):
        """Initialize the ProcessorBase instance.

        Args:
            job (smartElements.job.Job): The job instance that holds
                information about the job to ingest.

        """
        self.job = job
        self.media = job.media
        self.lock = LOCK

        if job.settings and job.settings.get(self.name):
            self.settings = job.settings.get(self.name)
        else:
            self.settings = self.DEFAULT_SETTINGS

        self._start = 0

        name = self.__class__.__name__
        module_path = "smartElements.ingest_processors.{}".format(name)
        self.logger = logging.getLogger(module_path)
        self.logger.setLevel(LEVEL)

    @property
    def name(self):
        """Get the name of the class.

        Returns:
            str: The name of the class.

        """
        return self.__class__.__name__

    def add_to_report(self, string):
        """Add the given string to the job report, add new line.

        Args:
            string (str): The string to add to the report.

        """
        self.job.add_to_job_report(string)

    def increase_progress(self, progress_path):
        """Update the progress in the given progress path by 1.

        Args:
            progress_path (str): Absolute path of the progress .json file
                to increase counter on.

        Returns:
            dict: The updated progress content.

        """
        if not os.path.isfile(progress_path):
            return

        with open(progress_path, "r") as file_:
            content = json.load(file_)
            current_new = content["current"] + 1
            content["current"] = current_new
        with open(progress_path, "w") as file_:
            json.dump(content, file_)

        return content

    def run(self):
        """Run the processor."""
        self._start = time.time()
        self.add_to_report("\n")
        self.add_to_report("{}".format(self.name))
        self.add_to_report("-" * len(self.name))

        if not self.settings.get("enable"):
            self.add_to_report("Skip as not being enabled.")
            return

        self.process()
        self.add_to_report(
            "Finished ingestion processor '{0}' in {1}s".format(
                self.name,
                time.time() - self._start
            )
        )

    @abc.abstractmethod
    def process(self):
        """Perform any action that this processor is supposed to execute."""
        pass
