"""Give the user a quick way to time offset media on import.

This import processor creates a dialog on importing media. This dialog lets
the user choose to start the media at a specific frame or to time offset the
media. It also contains a checkbox to let the import processor remember the
set values as long as the smartElements browser is open. Once it gets closed,
the remembered values will be flushed and the next time the user launches the
elements browser and inserts media from it into the DAG, the user will be
prompted with the dialog again.

"""

# Import built-in modules
import os

# Import local modules
from smartElements.import_processors.base_import_processor import BaseImportProcessor  # pylint: disable=line-too-long
from smartElements.mvc import helper
from smartElements.mvc.widgets import ComboBox
from smartElements.pys import QtCore, QtGui, QtWidgets

# Global view to keep it in memory so that it won't get garbage collected.
_VIEW = None

# Names of the environment variables under which we store the currently set
# up time offset values so that upcoming media insertions will get the same
# values without presenting the TimeOffset dialog again. These values will be
# stored as long as smartElement's element browser stays open. Once the user
# closes the elements browser, the setup gets flushed. When launching the
# browser again, the dialog re-appears.
_FRAME_MODE = "CRAGL_SMARTELEMENTS_FRAME_MODE"
_FRAME_VALUE = "CRAGL_SMARTELEMENTS_FRAME_VALUE"


class TimeOffsetView(QtWidgets.QDialog):
    """Panel to add time offsets to a node."""

    def __init__(self, node):
        """Initialize the TimeOffsetView instance.

        Args:
            node (nuke.Node): The node to set a time offset for.

        """
        super(TimeOffsetView, self).__init__()

        self.node = node

        self.setModal(True)
        self.setWindowFlags(QtCore.Qt.WindowStaysOnTopHint)
        self.setWindowTitle("TimeOffset {}".format(self.node.name()))

        self.create_widgets()
        self.create_layouts()

    def create_widgets(self):
        """Create widgets."""
        self.combo_frame_mode = ComboBox()
        self.combo_frame_mode.addItems(["start at", "offset"])
        self.input_frame_value = QtWidgets.QLineEdit()
        self.input_frame_value.setValidator(
            QtGui.QRegExpValidator(QtCore.QRegExp("[0-9]*")))
        self.check_remember = QtWidgets.QCheckBox("remember")

        self.push_update = QtWidgets.QPushButton("Update")
        self.push_update.setProperty("style", "action_button")
        self.push_skip = QtWidgets.QPushButton("Skip")

    def create_layouts(self):
        """Create layouts."""
        self.layout_main = QtWidgets.QVBoxLayout()
        self.layout_frame = QtWidgets.QHBoxLayout()
        self.layout_bottom = QtWidgets.QHBoxLayout()

        self.layout_frame.addWidget(self.combo_frame_mode)
        self.layout_frame.addWidget(self.input_frame_value)
        self.layout_frame.addWidget(self.check_remember)

        self.layout_bottom.addWidget(self.push_update)
        self.layout_bottom.addWidget(self.push_skip)

        self.layout_main.addLayout(self.layout_frame)
        self.layout_main.addLayout(self.layout_bottom)

        self.setLayout(self.layout_main)
        helper.set_style_sheet(self)

    def keyPressEvent(self, event):
        """Overwrite keyPressEvent to close window when pressing escape key.

        Args:
            event (QtCore.QEvent): The event instance that got triggered.

        """
        if event.key() == QtCore.Qt.Key_Escape:
            self.close()


class TimeOffsetController(object):
    """Drive a TimeOffsetView instance."""

    def __init__(self, view):
        """Initialize the TimeOffsetController instance.

        Args:
            view (smartElements.import_processors.time_offset.TimeOffsetView):
                The view to drive.

        """
        self.view = view

        self.create_signals()
        self.move_under_cursor()
        self.create_tooltips()

    def create_signals(self):
        """Create signals."""
        self.view.push_update.clicked.connect(lambda: self.update())
        self.view.push_skip.clicked.connect(lambda: self.view.reject())

    def update(self):
        """Update the node by the set inputs."""
        frame_mode = str(self.view.combo_frame_mode.currentText())
        frame_value = self.view.input_frame_value.text()
        if not frame_value:
            return

        self.view.node["frame_mode"].setValue(frame_mode)
        self.view.node["frame"].setValue(frame_value)

        if self.view.check_remember.isChecked():
            os.environ[_FRAME_MODE] = frame_mode
            os.environ[_FRAME_VALUE] = frame_value

        self.view.accept()

    def move_under_cursor(self):
        """Move given widget, respecting the current click position.

        Move with correctly offsetting from local click position relative to
        current widget.

        Args:
            widget_to_move (QtWidget.QWidget): Widget to move.
            click_pos (int, int): Current location of cursor relative to widget.
            event (QtCore.QEvent): Event instance.

        """
        xpos = QtGui.QCursor().pos().x() - 0.5 * self.view.width()
        ypos = QtGui.QCursor().pos().y() - 0.5 * self.view.height()
        self.view.move(xpos, ypos)

    def create_tooltips(self):
        """Create tooltips for all major widgets."""
        self.view.combo_frame_mode.setToolTip(
            "<b>frame mode</b>"
            "<br />"
            "Choose between starting the media at a specific frame or time "
            "offsetting the media to a specific amount of frames."
        )
        self.view.input_frame_value.setToolTip(
            "<b>frame value</b>"
            "<br />"
            "Enter the frame to start at or time offset to."
        )
        self.view.check_remember.setToolTip(
            "<b>remember</b>"
            "<br />"
            "If checked then we will remember the set up frame mode values "
            "and apply them to all upcoming media insertions as long as the "
            "elements browser stays open. "
        )
        self.view.push_update.setToolTip(
            "<b>update</b>"
            "<br />"
            "Click to apply the time offset values to the Read node."
        )
        self.view.push_skip.setToolTip(
            "<b>skip</b>"
            "<br />"
            "Click to cancel and don't apply any time offset to the Read node."
        )


class TimeOffset(BaseImportProcessor):
    """Count the usage of media when it is inserted."""

    def process(self):
        """Run the processor."""
        # This import processor makes only sense for image data. Exit when
        # we are handling 3d data or toolset data.
        if self.is_3d or self.is_toolset:
            return

        # In case the user checked the 'remember' checkbox, we store the
        # current setup in our environment. In this case use the values being
        # stored in our environment for all upcoming media insertions and skip
        # showing the TimeOffset dialog. This lasts until the user closes the
        # elements browser.
        envs = (
            os.environ.get(_FRAME_MODE),
            os.environ.get(_FRAME_VALUE)
        )
        if all(envs):
            self.nodes[0]["frame_mode"].setValue(os.environ[_FRAME_MODE])
            self.nodes[0]["frame"].setValue(os.environ[_FRAME_VALUE])
            return

        # Use a global in here, otherwise the view would be garbage collected
        # immediately as it is not used anywhere else.
        global _VIEW
        _VIEW = TimeOffsetView(self.nodes[0])
        _VIEW.raise_()
        _VIEW.show()

        TimeOffsetController(_VIEW)
