"""Public API to communicate to smartElements."""

# Import built-in modules
import getpass
import json
import logging
import os
import platform
import re
import shutil
import time

# Import local modules
from smartElements import templates
from smartElements.clogging import LEVEL
from smartElements.favorites import Favorites
from smartElements.settings import Settings
from smartElements import constants
from smartElements import paths
from smartElements import processor

LOGGER = logging.getLogger(__name__)
LOGGER.setLevel(LEVEL)


def get_stacks(absolute_paths=False):
    """Get list of stack names / absolute paths of all current stacks.

    Args:
        absolute_paths (bool, optional): If True return absolute paths,
            otherwise return just the stack names.

    Returns:
        :obj:`list` of :obj:`str`: Sequence of absolute paths of all current
            stacks.

    """
    settings = Settings()

    settings_stack_paths = settings.get("stacks")

    # Hide the default stack by env var.
    if constants.DEFAULT_STACK_HIDDEN:
        try:
            settings_stack_paths.remove(templates.DEFAULT_STACK)
        except ValueError:
            # ValueError stack not contained.
            pass

    env_stack_paths = [
        path for path in
        os.environ.get(constants.ENV_ADDITIONAL_STACKS, "").split(os.pathsep)
    ]

    stack_paths = settings_stack_paths + env_stack_paths
    stack_paths = [
        path for path in stack_paths
        if path
        and os.path.isdir(path)
    ]

    if absolute_paths:
        return stack_paths

    return [os.path.basename(path) for path in stack_paths]


def get_stack_path(stack_name):
    """Get the absolute path of the given stack name.

    Args:
        stack_name (str): The name of the stack to get the absolute path
            for.

    Returns:
        str: Absolute path of the stack with the given name.

    Raises:
        ValueError: When there is no such stack with the given name.

    """
    stack_paths = get_stacks(absolute_paths=True)
    for stack_path in stack_paths:
        if os.path.basename(stack_path.lower()) == stack_name.lower():
            return paths.sanitize(stack_path)

    raise ValueError(
        "No such stack '{0}' found. Choose from: {1} or "
        "add new stack using smartElements.api.add_stack.".format(
            stack_name, [os.path.basename(name) for name in get_stacks()])
    )


def get_lists(stack_name, absolute_paths=False):
    """Get all list names or absolute paths of given stack.

    Args:
        stack_name (str): The stack name to get all lists for.
        absolute_paths (boolean, If True): return absolute paths of lists
            in the given stack, otherwise return just the list names in the
            given stack.

    Returns:
        :obj:`list` of :obj:`str`: Sequence of lists names or absolute paths
            of lists in the given stack, depending on absolute_paths kwarg,
            sorted alphabetically.

    """
    stack_path = get_stack_path(stack_name)
    if not os.path.isdir(stack_path):
        return []

    names = sorted([
        name for name in os.listdir(stack_path)
        if os.path.isdir(os.path.join(stack_path, name))
        and not name.startswith(".")
    ], key=lambda x: x.lower())

    if not absolute_paths:
        return names

    return [
        paths.sanitize(os.path.join(stack_path, name))
        for name in names
    ]


def get_list_path(stack_name, list_name):
    """Get the absolute path of the given list name in given stack name.

    Args:
        stack_name (str): The name of the stack to get the given list for.
        list_name (str): The name of the list to get the absolute path for.

    Returns:
        str: Absolute path of the list with the given name in the given stack.

    Raises:
        ValueError: When there is no such stack with the given name or when
            there is no such list inside the given stack.

    """
    stack_path = get_stack_path(stack_name)
    for name in os.listdir(stack_path):
        if name.lower() == list_name.lower():
            return paths.sanitize(os.path.join(stack_path, name))

    raise ValueError(
        "No such list '{0}' found in stack '{1}'. Choose from: {2}. "
        "You can add a new list using smartElements.api.add_list.".format(
        list_name,
        stack_name,
        get_lists(stack_name)
    ))


def get_element(stack, list, name):
    """Get a particular element by given stack, list, name.

    Args:
        stack (str): The name of the stack the element lives in.
        list (str): The name of the list the element lives in.
        name (str): The name of the element.

    Returns:
        smartElements.api.Element: The element instance of the given stack,
            list, name.

    """
    return Element(stack, list, name)


def get_element_from_root(root):
    """Construct an element instance from given element root directory.

    Args:
        root (str): Absolute path of the element's root directory.

    Returns:
        smartElements.api.Element: The element instance from the given root
            directory.

    """
    root = root.replace("\\", "/")
    parts = root.split("/")

    return Element(parts[-3], parts[-2], parts[-1])


def get_element_from_path(path):
    """Get element from sequence path.

    This gets the element from the given path in case this is an element in
    smartElements.

    Args:
        path (str): Absolute path of the element to search in smartElements.

    Returns:
        smartElements.api.Element: The element that relates to the given path.

    Raises:
        ValueError: When the given path is not detected as element.

    """
    root = os.path.dirname(path)

    # Take version into account as the path could possibly be a versioned
    # element.
    version_substring = "/versions/v"
    if version_substring in root:
        root = root.split(version_substring)[0]

    return get_element_from_root(root)


def get_all_elements_in_stack(stack_name):
    """Get all elements in the given stack.

    Args:
        stack_name (str): The name of the stack to get all elements for.

    Returns:
        :obj:`list` of :obj:`smartElements.api.Element`: Sequence of all
            Element instances in the given stack.

    """
    elements = []
    for list_name in get_lists(stack_name):
        elements.extend(get_all_elements_in_list(stack_name, list_name))

    return elements


def get_all_elements_in_list(stack_name, list_name):
    """Get all elements in the given list.

    Args:
        stack_name (str): The name of the stack to look up.
        list_name (str): The name of the list to get all elements of.

    Returns:
        :obj:`list` of :obj:`str`: Sequence of absolute paths of all elements
            in the given list.

    """
    list_path = get_list_path(stack_name, list_name)

    elements = []
    for element_name in os.listdir(list_path):
        if element_name.startswith("."):
            continue
        if not os.path.isdir(os.path.join(list_path, element_name)):
            continue

        elements.append(get_element(stack_name, list_name, element_name))

    return elements


def get_elements_by_tags(stack_name, tags, match_strict=True):
    """Get list of elements using the given tags, using loose or strict match.

    Args:
        stack_name (str): The name of the stack the elements live in.
        tags (:obj:`list` of :obj:`str`): Sequence of tags an element must
            match.
        match_strict (bool, optional): If True return only elements that
            contain >all< of the given tags, otherwise, return all elements
            that contain at least one of the given tags.

    Returns:
        :obj:`list` of :obj:`smartElements.api.Element`: Sequence of
            elements that contain the given tags, either strict or loosely.

    """
    matching_elements = []

    for element in get_all_elements_in_stack(stack_name):
        element_tags = element.meta.get("tags") or []
        if match_strict:
            for tag in tags:
                if tag not in element_tags:
                    break
            else:
                matching_elements.append(element)
        else:
            for tag in tags:
                if tag in element_tags:
                    matching_elements.append(element)
                    break

    return matching_elements


def get_element_path(stack_name, list_name, element_name):
    """Get the absolute path of the element in the given stack, list, name.

    Args:
        stack (str): The name of the stack the element lives in.
        list (str): The name of the list the element lives in.
        name (str): The name of the element.

    Returns:
        str: Absolute path of the element in given stack, list, name.

    Raises:
        ValueError: When there is no such element with the given name in
            the given list.

    """
    list_path = get_list_path(stack_name, list_name)
    for name in os.listdir(list_path):
        if name.lower() == element_name.lower():
            return paths.sanitize(os.path.join(list_path, name))

    raise ValueError(
        "No such element '{0}' found in list '{1}' in stack '{2}'. Choose "
        "from these element names: {3}. Use smartElements.api.ingest to "
        "ingest a new element.".format(
        element_name,
        list_name,
        stack_name,
        [e.name for e in get_all_elements_in_list(stack_name, list_name)]
    ))


def add_stack(stack_path):
    """Add the given path as new stack, create it if not existing on disk.

    Args:
        stack_path (str): Absolute path of the folder to add as a new stack.

    """
    if stack_path in get_stacks():
        return

    paths.ensure_directory(stack_path)

    stacks = get_stacks()
    stacks.append(stack_path)

    settings = Settings()
    settings.save_setting("stacks", stacks)


def add_list(stack_name, list_name):
    """Add a new list with the given name to the given stack.

    Args:
        stack_name (str): Name of the stack to add a new list to.
        list_name (str): Name of the new list to create.

    """
    stack_path = get_stack_path(stack_name)
    list_path = os.path.join(stack_path, list_name)
    paths.ensure_directory(list_path)


def remove_stack(stack_name):
    """Remove the given stack from the settings, but doesn't delete it.

    Notes:
        This will not delete the stack or any physical file or folder, it
        will just remove the stack from the settings.

    """
    stacks = get_stacks()
    for stack_path in list(stacks):
        if os.path.basename(stack_path).lower() == stack_name.lower():
            stacks.remove(stack_path)

    settings = Settings()
    settings.save_setting("stacks", stacks)


def get_settings():
    """Get the whole smartElements settings dict.

    Returns:
        dict: The whole smartElements settings dict.

    """
    settings = Settings()
    return settings._settings


def save_setting(key, value):
    """Save the given settings key with the given value.

    Args:
        key (str): The name of the settings key to update.
        value (any): The new value to update to.

    """
    settings = Settings()
    settings.save_setting(key, value)


def scan_for_media(ingestion_paths):
    """Scan for media from given paths.

    This walks over the given paths and all sub directories to get all
    ingestable media from it.

    Args:
        ingestion_paths (str or :obj:`list` of :obj:`str`): One or multiple
            absolute paths (file or folder paths) to scan for all media.

    Returns:
        dict: All nested media from the dropped path(s) in the format:
            {
                <media basename>:
                    <Media instance of media1 relative from root>,
                <media basename>:
                    <Media instance of media2 relative from root>,
            }

    """
    return paths.scan_for_media(ingestion_paths)


def get_tags(stack_name):
    """Get all element tags of the given stack.

    Args:
        stack_name (str): Name of the stack to get all element tags for.

    Returns:
        :obj:`list` of :obj:`str`: Sequence of tags on all elements in the
            given stack, sorted alphabetically.

    """
    tags = []
    for element in get_all_elements_in_stack(stack_name):
        tags.extend(element.get_tags())

    return sorted(list(set(tags)), key=lambda x: x.lower())


def ingest(stack_name, list_name, paths, nuke_exe_path, ingestion_setup=None, num_threads=4):
    """Ingest paths to the given location.

    Args:
        stack_name (The name of the stack to ingest to.)
        list_name (The name of the list to ingest to.)
        paths (str or :obj:`list` of :obj:`str`): Path or sequence of paths to
            ingest. These paths can contain nested media which will be
            detected.
        nuke_exe_path (str): The absolute path of the Nuke executable to
            use for the ingestion (preview sequence rendering, metadata
            extraction, etc.).
        ingestion_setup (dict): The ingestion setup to use. If not given
            look it up from the local settings. This setup defines the
            configuration of the import processors.
        num_threads (int): Number of threads to use for concurrent ingestion.

    """
    # Perform lazy load so we only need to depend on nuke when ingesting.
    from smartElements import job

    # Set this env var so that he DataExtractor ingest processor can
    # react to it and use this as nuke executable.
    os.environ[constants.ENV_NUKE_EXE] = nuke_exe_path

    ingestion_setup = ingestion_setup or Settings().get("ingestion_setup")
    ingestion_setup["general"] = {
        "list": get_list_path(stack_name, list_name)
    }

    ingest_processor = processor.Processor(num_threads=num_threads)

    if isinstance(paths, str):
        paths = [paths]

    for media in scan_for_media(paths).values():
        LOGGER.info("Adding to ingest queue: %s" % media.path)
        ingest_processor.add_job(job.Job(media, ingestion_setup, logger=LOGGER))


class Element(object):
    """Represent an element as object."""

    def __init__(self, stack, list, name):
        """Initialize the Element instance.

        Args:
            stack (str): The name of the stack the element lives in.
            list (str): The name of the list the element lives in.
            name (str): The name of the element.

        """
        self.stack = stack
        self.list = list
        self.name = name

        self.root_path = get_element_path(stack, list, name)
        self.meta_path = paths.get_meta_path(self.root_path)

    def __repr__(self):
        """Create an instance representation.

        Returns:
            str: An element representation.

        """
        return "smartElements.api.Element('{0}', '{1}', '{2}')".format(
            self.stack,
            self.list,
            self.name
        )

    def __str__(self):
        """Create a human-readable instance representation.

        Returns:
            str: A human-readable element representation.

        """
        return (
            "smartElements.api.Element: {0}/{1}/{2}".format(
                self.stack,
                self.list,
                self.name
            )
        )

    @property
    def meta(self):
        """Get all metadata for the given element.

        Returns:
            dict: The metadata for the given element.

        """
        if not os.path.isfile(self.meta_path):
            return {}

        with open(self.meta_path, "r") as file_:
            return json.load(file_)

    @property
    def sequence(self):
        """Get element's preview jpg sequence.

        Returns:
            :obj:`list` of :obj:`str`: Sequence of absolute paths of the
                element's preview .jpg sequence.

        """
        preview_root = os.path.join(self.root_path, ".meta", "seq")
        if not os.path.isdir(preview_root):
            return []

        return sorted([
            os.path.join(preview_root, name)
            for name in os.listdir(preview_root)
            if name.endswith(".jpg")
        ])

    def set_thumbnail(self, index=None, first=False, middle=False, end=False, auto=False):
        """Set a given thumbnail for the element.

        This copies a frame from the preview sequence to 'preview.jpg', so
        this assumes a preview sequence exists.

        Args:
            index (int, optional): If set, use a particular index of the
                element's sequence.
            first (bool, optional): If True use the first frame as thumbnail.
            middle (bool, optional): If True use the middle frame as thumbnail.
            end (bool, optional): If True use the end frame as thumbnail.
            auto (bool, optional): If True use the largest frame that contains
                the most image information, in most cases (but not always)
                representing the most interesting and most representative
                frame in the sequence.

        Returns:
            str: Absolute path of preview frame that was set as thumbnail.

        """
        if index is not None:
            try:
                path = self.sequence[index]
            except IndexError:
                raise IndexError(
                    "Index out of element's sequence bounds. Choose index "
                    "between 0 - {0}".format(len(self.sequence) - 1)
                )

        elif first:
            path = self.sequence[0]

        elif middle:
            middle_index = int((len(self.sequence) - 1) / 2)
            path = self.sequence[middle_index]

        elif end:
            path = self.sequence[-1]

        elif auto:
            largest_frame = self.sequence[0]
            for path in self.sequence[1:]:
                if os.path.getsize(path) > os.path.getsize(largest_frame):
                    largest_frame = path
            path = largest_frame

        dest = os.path.join(self.root_path, ".meta", "preview.jpg")
        shutil.copy(path, dest)

        return path

    def get_preview_path(self):
        """Get element's preview path or default preview path as fallback.

        Returns:
            str: Absolute path to the element's preview path if it exists or
                the default preview image path when it does not exist.

        """
        return paths.get_media_preview_path(self.root_path)

    def delete(self):
        """Physically delete the element from disk.

        Notes:
            WARNING: There is no undo! Handle with caution!

        """
        shutil.rmtree(self.root_path)

    def deprecate(self, reason):
        """Mark the element as deprecated.

        Args:
            reason (str): The reason to display with the element to inform
                other users why the element shouldn't be used anymore.

        """
        meta = self.meta
        meta["deprecated"] = {
            "reason": reason,
            "user": getpass.getuser(),
            "machine": platform.node(),
            "date": time.strftime("%Y/%m/%d"),
        }
        with open(self.meta_path, "w") as file_:
            json.dump(meta, file_)

    def undeprecate(self):
        """Remove deprecation from element."""
        meta = self.meta
        meta.pop("deprecated", None)
        with open(self.meta_path, "w") as file_:
            json.dump(meta, file_)

    def set_favorite(self, favorite_index):
        """Set the favorite index for this element.

        Args:
            favorite_index (int): The favorite index to set from 1-5. Use
                0 to remove favorite index.

        Raises:
            ValueError: When the given favorite_index is not between 0-5.

        """
        list_path = os.path.dirname(self.root_path)
        favorites = Favorites()

        if favorite_index == 0:
            favorites.remove(list_path, self.root_path)
        elif favorite_index in range(1, 6):
            favorites.save(list_path, self.root_path, favorite_index)
        else:
            raise ValueError("Invalid favorite index. Choose between 0-5.")

    def set_comment(self, comment):
        """Set the comment on the element.

        Args:
            comment (str): The comment to use on the element.

        """
        meta = self.meta
        meta["comment"] = comment
        with open(self.meta_path, "w") as file_:
            json.dump(meta, file_)

    def get_tags(self):
        """Get all tags of the element.

        Returns:
            :obj:`list` of :obj:`str`: Sequence of tags of the element.

        """
        return sorted(self.meta.get("tags") or [])

    def add_tag(self, tag_names):
        """Add the given tag names to the element.

        Notes:
            For convenience, the user can either pass in a string or a list
            of strings so that they just need to call this once.

        Args:
            tag_names (str|:obj:`list` of :obj:`str`): The tag name or several
                tag names to add to the element.

        """
        meta = self.meta
        tags = meta.get("tags") or []

        if isinstance(tag_names, str):
            tags.append(tag_names)
        elif isinstance(tag_names, list):
            tags.extend(tag_names)

        # Cleanups: Only string tags, remove whitespace, no duplicates.
        tags = [tag for tag in tags if isinstance(tag, str)]
        tags = [tag.strip() for tag in tags]
        tags = list(set(tags))
        meta["tags"] = tags

        with open(self.meta_path, "w") as file_:
            json.dump(meta, file_)

    def remove_tag(self, tag_name):
        """Remove the given tag from the element."""
        if not isinstance(tag_name, str):
            raise ValueError("Please specify a string type.")

        meta = self.meta
        tags = meta.get("tags") or []

        try:
            tags.remove(tag_name)
        except ValueError:
            # ValueError: Tag name not included in tags.
            return

        meta["tags"] = tags

        with open(self.meta_path, "w") as file_:
            json.dump(meta, file_)

    def reveal(self):
        """Reveal the given element in the explorer."""
        paths.reveal(self.root_path)

    def set_color(self, hex_color_string):
        """Assign the given hex color string.

        Args:
            hex_color_string (str): The hex color string to assign, with or
            without hash character. Example: #ff0000, #339933, #AABBCC.

        Raises:
            ValueError: When not given a string as hex color string or if
                the given value does not follow hex string notation.

        """
        examples = "#ff0000, #339933, #AABBCC."

        if not isinstance(hex_color_string, str):
            raise ValueError(
                "Please provide a hex color string in the format: "
                "{0}".format(examples)
            )

        # Ensure hash character in the beginning.
        hex_color_string = hex_color_string.replace("#", "")
        hex_color_string = "#{0}".format(hex_color_string).lower()

        # https://regex101.com/r/KKi0z6/1
        pattern = r"\#[0-9a-f]{6}"
        if not re.match(pattern, hex_color_string):
            raise ValueError(
                "Please provide a hex string in the format: {0}".format(
                    examples
                )
            )

        settings = Settings()
        element_colors = settings.get("element_colors")
        element_colors[self.root_path] = hex_color_string
        settings.save_setting("element_colors", element_colors)

    def unset_color(self):
        """Remove any custom color from the element."""
        settings = Settings()
        element_colors = settings.get("element_colors")
        element_colors.pop(self.root_path, None)
        settings.save_setting("element_colors", element_colors)

    def add_metadata(self, key, value):
        """Add or update a metadata key-value pair.

        Args:
            key (str): The name of the key to add or update.
            value (any): The value to assign to the metadata key.

        """
        meta = self.meta
        meta["meta"].update({key: value})
        self.meta["meta"] = meta
        with open(self.meta_path, "w") as file_:
            json.dump(meta, file_)

    def remove_metadata(self, key):
        """Remove given key from the metadata.

        Args:
            key (str): The name of the key to remove from the metadata.

        """
        meta = self.meta
        meta["meta"].pop(key, None)
        self.meta["meta"] = meta
        with open(self.meta_path, "w") as file_:
            json.dump(meta, file_)

    def rename(self, new_name):
        """Rename the element.

        Notes:
            In order to preserve consistency and not risking to break any
            working files that use this element, this will only update the
            name in the metadata, but will not rename any physical files
            on disk. Note also that the element ID will then still stay the
            same, i.e. the name on disk and this is how you still address
            it, not by then new name. So this controls just how the element's
            name is displayed in the smartElements UI, because it reads the
            'name' metadata and uses that as the displayable name.

        Args:
            new_name (str): The new name to use for the element.

        """
        meta = self.meta
        meta["name"] = new_name
        with open(self.meta_path, "w") as file_:
            json.dump(meta, file_)

    def add_additional_files(self, files_and_folder_paths, replace=False):
        """Add additional files and folder to the element.

        Args:
            add_additional_files (:obj:`list` of :obj:`str`): Sequence of
                file and folder paths to copy to the element. Skip any non
                existing paths.
            replace (bool, optional): If True replace already existing folders
                or files.

        Raises:
            ValueError: When not specifying a str or list of paths or when
                one of the provided values in the path list is not a path.

        """
        if isinstance(files_and_folder_paths, str):
            files_and_folder_paths = [files_and_folder_paths]
        elif not isinstance(files_and_folder_paths, list):
            raise ValueError(
                "Specify a single file/folder path or a list of file/folder "
                "paths."
            )

        data_root = paths.get_additional_files_root(self.root_path)
        paths.ensure_directory(data_root)

        for source_path in files_and_folder_paths:
            if not isinstance(source_path, str):
                raise ValueError(
                    "Cannot copy: {0}. Please provide a path.".format(
                        source_path
                    )
                )

            dest = os.path.join(data_root, os.path.basename(source_path))

            if not os.path.exists(source_path):
                print("No such existing file or folder: '{0}'".format(source_path))
                continue

            if replace:
                if os.path.isfile(dest):
                    os.remove(dest)
                if os.path.isdir(dest):
                    shutil.rmtree(dest)

            if os.path.isfile(source_path):
                shutil.copy2(source_path, dest)

            elif os.path.isdir(source_path):
                shutil.copytree(source_path, dest)
