"""Create preview frames for the media."""

# Import built-in modules
import json
import os
import sys

# Import third-party modules
import nuke

# In order to be able to import local modules from smartElements we must add
# the smartElement's packages root to our sys path. Only then are we able to
# import local modules.
PACKAGE_ROOT = os.path.abspath(os.path.join(__file__, "..", "..", ".."))
sys.path.append(PACKAGE_ROOT)

# Import local modules
from smartElements import api
from smartElements import paths
from smartElements.constants import VIDEO_EXTENSIONS


def switch_to_aces():
    """Switch to ACES color space."""
    nuke.root()["colorManagement"].setValue("OCIO")
    nuke.root()["OCIO_config"].setValue("aces_1.2")


def main(source, dest, width, start, end, thumbnail_frame):
    """Main entry point to create preview for media.

    Args:
        source (str): Absolute path of source media to render preview for.
        dest (str): Absolute path to write preview to.
        width (int): The width to reformat to. Will keep aspect ratio to
            use an automatic height.
        start (int): The first frame to render.
        end (int): The last frame to render.
        thumbnail_frame (str): The thumbnail frame to render. Choose from
            ("start", "middle", "end", "auto"). If not using any of these we
            fall back to "start".

    """
    # Make sure we sanitize all paths as Nuke does not understand backward
    # slashes.
    source = source.replace("\\", "/")
    dest = dest.replace("\\", "/")

    if not os.path.isdir(dest):
        os.makedirs(dest)

    read = nuke.createNode("Read")

    # Set up the path and frame range values to render. We must distinguish
    # between an image sequence and a provided video file. If we are handling
    # an image sequence then use the provided ``start``  and ``end`` value.
    # However, when handling a video file, we cannot use these values as they
    # don't reflect the correct frame range. We must then use the video's
    # native frame range which we get by using read["file"].fromUserText.
    extension = os.path.splitext(source)[1]
    if extension in VIDEO_EXTENSIONS:
        # Using setValue will not work as Nuke will not detect the source's
        # frame range. We need fromUserText so that Nuke will set up the
        # frame range correctly. For additional information see:
        # https://community.foundry.com/discuss/post/891640
        # Using fromUserText will both set the file knob value and the frame
        # range.
        read["file"].fromUserText(source)
        start = read["first"].value()
        end = read["last"].value()
    else:
        read["file"].setValue(source)
        for name in ("first", "origfirst"):
            read[name].setValue(start)
        for name in ("last", "origlast"):
            read[name].setValue(end)

    # By default, we render using Nuke's colorspace. If the Read node has an
    # error this is most probably due to an invalid colorspace. Switch to ACES.
    if read.hasError():
        switch_to_aces()

    # Create Reformat.
    # We always create a 16/9 image and use a letter box or pillar box when
    # the media does not fit this aspect ratio.
    reformat = nuke.createNode("Reformat")
    reformat.setXYpos(read.xpos(), read.ypos() + 100)
    reformat["type"].setValue("to box")
    reformat["box_width"].setValue(width)
    reformat["box_fixed"].setValue(True)
    reformat["box_height"].setExpression("int(box_width / 1.777777777)")
    reformat["black_outside"].setValue(True)
    reformat["filter"].setValue("Sinc4")
    if read.width() > read.height():
        reformat["resize"].setValue("width")
    else:
        reformat["resize"].setValue("height")

    # Create Write node for output and render range.
    write_seq = nuke.createNode("Write")
    write_seq.setXYpos(reformat.xpos(), reformat.ypos() + 100)
    counter = "#" * len(str(end))
    render_path = os.path.join(dest, "{}.jpg".format(counter))
    write_seq["file"].setValue(render_path.replace("\\", "/"))
    write_seq["file_type"].setValue("jpg")

    # Save .nk file that created this preview in element's meta directory.
    script_path = os.path.join(os.path.dirname(dest), "preview.nk")
    nuke.scriptSave(script_path)

    # Add after frame render callbacks to keep track of render progress.
    nuke.addAfterFrameRender(increase_progress)

    # Execute the seq rendering.
    nuke.execute(write_seq, start, end)

    # Create thumbnail.
    element_root = os.path.join(os.path.dirname(dest), "..")
    element_root = os.path.abspath(element_root)
    element = api.get_element_from_root(element_root)
    first = True if thumbnail_frame in ("first", "start") else False
    middle = True if thumbnail_frame == "middle" else False
    end = True if thumbnail_frame == "end" else False
    auto = True if thumbnail_frame == "auto" else False
    element.set_thumbnail(first=first, middle=middle, end=end, auto=auto)


def increase_progress():
    """Increase the render progress counter by 1.

    Args:
        job (smartMedia.job.Job): The job instance that holds data about
            frame range for progress calculation and the path to write to.

    """
    if not os.path.isfile(progress_path):
        return

    with open(progress_path, "r") as file_:
        content = json.load(file_)
        current_new = content["current"] + 1
        content["current"] = current_new
    with open(progress_path, "w") as file_:
        json.dump(content, file_)


if __name__ == "__main__":
    source = sys.argv[1]
    dst = sys.argv[2]
    width = int(sys.argv[3].replace("width=", ""))
    start = int(sys.argv[4].replace("start=", ""))
    end = int(sys.argv[5].replace("end=", ""))
    thumbnail_frame = sys.argv[6].replace("thumbnail_frame=", "")
    element_root = os.path.dirname(os.path.dirname(dst))
    progress_path = paths.get_element_progress_path(element_root)
    paths.ensure_directory(os.path.dirname(progress_path))

    main(source, dst, width, start, end, thumbnail_frame)
