"""Copy all media files to destination."""

# Import built-in modules
import os
import shutil

# Import local modules
from smartElements.ingest_processors.base_ingest_processor import BaseIngestProcessor
from smartElements import paths


class Copier(BaseIngestProcessor):
    """Copy all media files to destination."""

    def __init__(self, job):
        """Initialize the Copier instance.

        Args:
            job (smartElements.job.Job): The job instance that holds
                information about the job to ingest.

        """
        super(Copier, self).__init__(job)

        self.update_copy_mode_via_environment()

    def update_copy_mode_via_environment(self):
        """Update copy mode when set in environment."""
        env_hardcopy = os.environ.get("CRAGL_SMARTELEMENTS_INGEST_HARDCOPY", "")
        if env_hardcopy.lower() == "true":
            self.settings["enable"] = True
            self.logger.info(
                "Enforced hard copy via environment variable "
                "'CRAGL_SMARTELEMENTS_INGEST_HARDCOPY'"
            )

        env_softcopy = os.environ.get("CRAGL_SMARTELEMENTS_INGEST_SOFTCOPY",
                                      "")
        if env_softcopy.lower() == "true":
            self.settings["enable"] = False
            self.logger.info(
                "Enforced soft copy via environment variable "
                "'CRAGL_SMARTELEMENTS_INGEST_SOFTCOPY'"
            )

    def __init__(self, job):
        """Initialize the Copier instance.

        Args:
            job (smartElements.job.Job): The job instance that holds
                information about the job to ingest.

        """
        super(Copier, self).__init__(job)

        self.update_copy_mode_via_environment()

    def update_copy_mode_via_environment(self):
        """Update copy mode when set in environment."""
        env_hardcopy = os.environ.get("CRAGL_SMARTELEMENTS_INGEST_HARDCOPY", "")
        if env_hardcopy.lower() == "true":
            self.settings["enable"] = True
            self.logger.info(
                "Enforced hard copy via environment variable "
                "'CRAGL_SMARTELEMENTS_INGEST_HARDCOPY'"
            )

        env_softcopy = os.environ.get("CRAGL_SMARTELEMENTS_INGEST_SOFTCOPY",
                                      "")
        if env_softcopy.lower() == "true":
            self.settings["enable"] = False
            self.logger.info(
                "Enforced soft copy via environment variable "
                "'CRAGL_SMARTELEMENTS_INGEST_SOFTCOPY'"
            )

    def process(self):
        """Run the copy process."""
        parent_dst = paths.sanitize(self.job.parent_dst)

        self.logger.debug("Copying data to: %s", parent_dst)
        paths.ensure_directory(parent_dst)

        progress_path = paths.get_element_progress_path(parent_dst)

        for index, path in enumerate(self.media.files, 1):
            path = paths.sanitize(path)
            dst = os.path.join(parent_dst, os.path.basename(path))
            dst = paths.sanitize(dst)

            if os.path.isfile(dst) and self.settings["overwrite"]:
                self.add_to_report("Removing {}".format(dst))
                try:
                    os.remove(dst)
                except OSError:
                    # In case the file does not exist anymore or if it is
                    # locked then we cannot remove it. In that case pass
                    # and continue.
                    pass

            # Copy file.
            self.add_to_report("Copying from {} to {}".format(path, dst))
            shutil.copy(path, dst)

            # Copy support RMD files if present.
            parent = os.path.dirname(path)
            rmd_files = [
                name for name in os.listdir(parent)
                if name.lower().endswith("rmd")
            ]
            if rmd_files:
                for name in rmd_files:
                    rmd_src = os.path.join(parent, name)
                    rmd_dst = os.path.join(parent_dst, name)
                    self.add_to_report(
                        "Copying RMD file from {} to {}".format(
                            rmd_src, rmd_dst)
                    )
                    shutil.copy(rmd_src, rmd_dst)

            self.increase_progress(progress_path)
