"""Copy inserted media relative to the working file and remap node's path.

This import processor will copy the inserted media relative to the currently
opened working file and remap the created node's `file` node to read this new
copy. Both processes happen in a parallel thread so that the user does not get
blocked. First it will assemble the destination directory to copy the inserted
media to. This path is based on the location of the currently opened Nuke
working file and the relative location that is set in the settings key
`media_path_relative_to_working_file` for the `CopyToWorkingFile` settings. It
will then ensure this directory exists, copy the media to this location and
update the node to use this location instead.

"""

# Import built-in modules
import os
import shutil
import threading

# Import local modules
from smartElements.import_processors.base_import_processor import BaseImportProcessor  # pylint: disable=line-too-long


class CopyToWorkingFile(BaseImportProcessor):
    """Copy the media to a folder relative to the working file."""

    def _assemble_target_root(self):
        """Assemble the root to copy media to.

        Returns:
            str: Absolute root directory to copy media to.

        """
        import nuke

        working_file_path = nuke.root().name()
        if working_file_path in ("Root", "root"):
            return ""

        working_file_root = os.path.dirname(working_file_path)
        footage_root = os.path.join(
            working_file_root,
            self.settings["media_path_relative_to_working_file"]
        )
        footage_root = os.path.abspath(footage_root)
        return os.path.join(footage_root, self.media.base).replace("\\", "/")

    def _ensure_root(self, path):
        """Ensure the destination root directory exists.

        Args:
            path (str): Absolute folder path to create.

        """
        if os.path.isdir(path):
            return

        try:
            os.makedirs(path)
        except OSError as error:
            self.logger.error(str(error))
            return

    def _copy_media(self, dest_root):
        """Copy all media files to given destination root.

        Args:
            dest_root (str): Absolute path to directory to copy data to.

        """
        for path in self.media.files:
            dest = os.path.join(dest_root, os.path.basename(path))
            if os.path.isfile(dest) and self.settings["overwrite"]:
                continue

            self.logger.debug("Copying from %s to %s", path, dest)
            shutil.copy(path, dest)

    def _remap_node(self, dest_root):
        """Remap the node to the copied media."""
        import nuke

        dest = os.path.join(dest_root, os.path.basename(self.media.files[0]))
        # When inserting media into the DAG, we store all nodes that get
        # generated. In the case of an image media, this created -one- Read
        # node. In the case of a 3D media, this created -one- ReadGeo node.
        # For both node we are interested in the `file` knob.
        # That means the node that we are interested in is in index 0.
        nuke.executeInMainThread(lambda: self.nodes[0]["file"].setValue(dest))

    def copy_and_remap(self, dest_root):
        self.logger.debug("Copying media to %s", dest_root)
        self._copy_media(dest_root)

        self.logger.debug("Remapping '%s' to reference copied media",
                          self.nodes[0].name())
        self._remap_node(dest_root)

    def process(self):
        """Run the processor."""
        dest_root = self._assemble_target_root()
        # Return early when the current working file is unsaved as we don't
        # know where we should copy the media to.
        if not dest_root:
            self.logger.debug("Working file is not saved. Will not copy data.")
            return

        # While it makes sense to copy image- and 3D data, it does not make
        # too much sense to copy any toolset because the nodes will become
        # part of the actual working file and we don't need the actual toolset
        # being copied over next to the working file. If we are handling a
        # toolset here let's return early and not copy anything over.
        if self.is_toolset:
            return

        self._ensure_root(dest_root)

        # If the destination folder could not be created for whatever reason
        # we must stop here as we won't be able to copy over the data. The
        # reason why the folder could not be created will already be logged
        # at this stage.
        if not os.path.isdir(dest_root):
            return

        thread = threading.Thread(
            target=self.copy_and_remap,
            args=(dest_root,)
        )

        thread.start()
