"""Abstract import base processor that sets up the needed structure.

The BaseImportProcessor set up the base layer for an import processor that
can be used for all kinds of import processes. We set up a few things that
you can use to your advantage so that writing custom import processors should
be easy to accomplish. When inserting media into the DAG (either by dropping
media from smartElement's element browser into the DAG or by right clicking
media and choosing 'Insert' from the context menu) then all import processors
are triggered. When creating a new import processor, we advise to inherit it
from the BaseImportProcessor that is defined in here. Afterwards, make sure
to register it in smartElements.import_processors.__init__.py by calling:

    osl.register_import_processor(<NAME OF IMPORT PROCESSOR CLASS>)

Each import processor must implement a `process` function. This function
defines the custom logic to run when triggering the import processor. Your
custom import processor will be set up with multiple attributes automatically
that you can use to your advantage:
    - path
    - nodes
    - is_toolset
    - is_3d
    - is_image_data

Please refer to the BaseImportProcessor's doc string for further information on
these. In addition, the BaseImportProcessor provides other properties that you
might need in your process method. This base class provides these so that you
don't need to invest time creating these yourself:
    - name
    - media_root
    - meta_json

Please refer to the properties doc strings as found in the BaseImportProcessor
class below.

"""
# Import built-in modules
import abc
import logging
import os
import time

# Import local modules
from smartElements import media
from smartElements.clogging import LEVEL
from smartElements.settings import Settings


class BaseImportProcessor(object):
    """Abstract base of a processor that runs when importing media into DAG."""

    __metaclass__ = abc.ABCMeta

    # The default settings to apply for a processor if not otherwise specified
    # in the smartElement's import processor settings.
    DEFAULT_SETTINGS = {
        "enable": True,
    }

    def __init__(self, path, nodes, is_toolset, is_3d, is_image_data):
        """Initialize the BaseImportProcessor instance.

        Notes:
            All the attributes will be given by smartElements automatically
            when importing media into the DAG so you have multiple attributes
            that you can refer to if needed.

        Args:
            path (str): Absolute path of the first file of media. This will
                automatically be set for you when dropping media into your DAG.
            nodes (:obj:`list` of :obj:`nuke.Node`): The nodes that got created
                when bringing in the media into the DAG. When inserting an
                image type media or 3D data, the list's length will probably
                be 1, otherwise it will contain all the inserted nodes when
                having brought a toolset into the DAG.
            is_toolset (bool), If True we are handling media that is a toolset,
                meaning we dropped a toolset from smartElements into the DAG.
                Use this flag to do certain things in the process method that
                should only be done for toolsets, or to exit early if needed.
            is_3d (bool), If True we are handling media that is 3D data,
                meaning we dropped 3D data from smartElements into the DAG.
                Use this flag to do certain things in the process method that
                should only be done for 3d data, or to exit early if needed.
            is_image_data (bool), If True we are handling media that is a any
                image media (any image or a video), meaning we dropped an image
                type media from smartElements into the DAG.
                Use this flag to do certain things in the process method that
                should only be done for image data, or to exit early if needed.

        """
        self.path = path
        self.nodes = nodes
        self.is_toolset = is_toolset
        self.is_3d = is_3d
        self.is_image_data = is_image_data

        self.media = media.Media(path)
        import_settings = Settings().load_settings()["import_processors"]
        self.settings = import_settings.get(self.name, self.DEFAULT_SETTINGS)

        self.logger = logging.getLogger(self.__class__.__name__)
        self.logger.setLevel(LEVEL)

    @property
    def name(self):
        """Get the name of the class.

        Returns:
            str: The name of the class.

        """
        return self.__class__.__name__

    @property
    def media_root(self):
        """Get the absolute path of the media's root directory.

        Returns:
            str: Absolute path of the media's root directory.

        """
        return os.path.dirname(self.path)

    @property
    def meta_json(self):
        """Get the absolute path to the media's meta.json file.

        Returns:
            str: Absolute path to the media's meta.json file.

        """
        return os.path.join(self.media_root, ".meta", "meta.json")

    def run(self):
        """Run the processor."""
        self._start = time.time()
        self.logger.debug("Running custom import processor '%s'", self.name)

        if self.settings.get("enable"):
            try:
                self.process()
            # We want to explicitly catch anything that went wrong here and
            # log it instead of stopping the import chain here, otherwise this
            # would stop importing upcoming elements which we want to avoid
            # at any costs.
            except Exception as error:  # pylint: disable=broad-except
                self.logger.exception(str(error))

        else:
            self.logger.debug("'%s' is disabled. Skip processing", self.name)
        duration = time.time() - self._start
        self.logger.debug("'%s' finished in %ss", self.name, duration)

    @abc.abstractmethod
    def process(self):
        """Perform any action that this processor is supposed to execute."""
        pass
