"""Add element to used-in statistics.

This import processor will add the current working file to the used-in statistics for that element,
so that anybody can see where elements have been used. This helps e.g. in copying over setup from
working files where elements have been used.

"""

# Import built-in modules
import copy
import json
import re

# Import local modules
from smartElements import api
from smartElements.import_processors.base_import_processor import BaseImportProcessor


class AddToUsedStatistics(BaseImportProcessor):
    """Add element to used-in statistics."""

    USED_IN = "used_in"

    @staticmethod
    def make_version_arbitrary(path):
        """Make the version arbitrary/masked for given path.

        Usage:
            >>> make_version_arbitrary("/path/to/workfile_v001.nk")
            /path/to/workfile_v*.nk

            >>> make_version_arbitrary("/path/to/workfile.v001.nk")
            /path/to/workfile_v*.nk

            >>> make_version_arbitrary("/path/to/workfile.nk")
            /path/to/workfile.nk

        Returns:
            str: Given path with an arbitrary/masked version number.

        """
        # https://regex101.com/r/ZLL7ch/1
        pattern = re.compile(r".*[\._]+[vV]+(\d+).nk")

        def _sub(m):
            full = m.group(0)
            full = full.replace(m.group(1), "*")
            return full

        return pattern.sub(_sub, path)

    def process(self):
        """Run the processor."""
        self.logger.debug("Adding element to used-in statistics %s: ", self.media.base)

        try:
            import nuke
        except ImportError:
            return

        current_workfile = nuke.root().name()
        if current_workfile == "Root":
            return

        element = api.get_element_from_path(self.path)
        meta = copy.deepcopy(element.meta)

        if self.USED_IN not in meta:
            meta[self.USED_IN] = []

        workfile_arbitrary_version = self.make_version_arbitrary(current_workfile)
        if workfile_arbitrary_version in meta[self.USED_IN]:
            return

        meta[self.USED_IN].insert(0, workfile_arbitrary_version)

        with open(element.meta_path, "w") as file_:
            json.dump(meta, file_, indent=4)
