"""Extract data from media."""

# Import built-in modules
import json
import os
import sys

# Import third-party modules
import nuke

# In order to be able to import local modules from smartElements we must add
# the smartElement's packages root to our sys path. Only then are we able to
# import local modules.
PACKAGE_ROOT = os.path.abspath(os.path.join(__file__, "..", "..", ".."))
sys.path.append(PACKAGE_ROOT)

# Import local modules
from smartElements import dtime
from smartElements import media
from smartElements.constants import INGESTABLE_EXTENSIONS_FOOTAGE
from smartElements.constants import VIDEO_EXTENSIONS


def main(source, element_root):
    """Extract data from media.

    Notes:
        We extract data only from footage sources (image sequences/ videos).
        Extracting data from .nk files or 3D data does not make too much sense.

    Args:
        source (str): Absolute path of the source to extract data from.
        element_root (str): Absolute path or element directory where the
            element will live in, or if this is a soft copy, where at least
            all metadata will live in.

    """
    source = source.replace("\\", "/")
    data = {}

    # Extract additional data from footage if this is a file sequence or video.
    extension = os.path.splitext(source)[1]
    if extension in INGESTABLE_EXTENSIONS_FOOTAGE:
        read = nuke.createNode("Read")

        # Extract the correct frame range of media. We must distinguish
        # between a mov and other file types.
        if extension in VIDEO_EXTENSIONS:
            # Using setValue on video files will not work as Nuke will not
            # detect the source's frame range. We need fromUserText so that
            # Nuke will set up the frame range correctly. For additional
            # information see:
            # https://community.foundry.com/discuss/post/891640
            # Using fromUserText will both set the file knob value and the
            # frame range.
            read["file"].fromUserText(source)
            # Need to add an additional 1 as the first frame counts, too.
            number_frames = read["last"].value() - read["first"].value() + 1
            first_frame = read["first"].value()
            last_frame = read["last"].value()
        else:
            read["file"].setValue(source)
            media_ = media.Media(source)
            number_frames = len(media_.files)
            first_frame = media_.first
            last_frame = media_.last

        data["meta"] = read.metadata()
        data["meta"]["date_ingested"] = dtime.current_date_time()
        data["width"] = read.width()
        data["height"] = read.height()
        data["frames"] = number_frames
        data["first_frame"] = first_frame
        data["last_frame"] = last_frame

        # Save .nk file that created this preview in element's meta directory.
        script_path = os.path.join(
            element_root,
            ".meta",
            "metadata.nk"
        )
        if not os.path.isdir(os.path.dirname(script_path)):
            os.makedirs(os.path.dirname(script_path))

        nuke.scriptSave(script_path)

    sys.stdout.flush()

    # Wrap data inside keywords that we split away later.
    sys.stdout.write("DATA={}ENDDATA".format(json.dumps(data)))

    sys.stdout.flush()


if __name__ == "__main__":
    source = sys.argv[1]
    element_root = sys.argv[2]

    main(source, element_root)
