"""Send selected media to RV.

The media of the selected elements will be send to the RV media player. In
order to make that happen, you will of course need to tell smartElements where
the RV executable is located. This gets done by setting the environment
variable CRAGL_SMARTELEMENTS_RV_EXECUTABLE to the executable path of RV.

"""

# Import built-in modules
import os
import re
import subprocess

# Import local modules
from smartElements.media_commands.base_media_command import BaseMediaCommand


class SendToRV(BaseMediaCommand):
    """Send selected media to DJV."""

    EXE_PATH = "CRAGL_SMARTELEMENTS_RV_EXECUTABLE"

    def get_executable(self):
        """Get the RV executable.

        Returns:
            str: Absolute path to the RV executable.

        Raises:
            EnvironmentError: When the environment variable
                CRAGL_SMARTELEMENTS_RV_EXECUTABLE which points to the RV
                executable is not set.
            ValueError: When the executable does not point to a file.

        """
        exe_path = os.environ.get(self.EXE_PATH)
        if not exe_path:
            raise EnvironmentError(
                "Environment variable '{}' which should point to the RV "
                "executable is not set. Please make sure to set it."
                "".format(self.EXE_PATH)
            )

        if not os.path.isfile(exe_path):
            raise ValueError(
                "Environment variable '{}' does not point to a file. Make "
                "sure it points to the RV executable.".format(self.EXE_PATH)
            )

        return exe_path

    def assemble_command(self, media):
        """Assemble the command to run in order to send media to DJV.

        Notes:
            RV expects the following notation when sending media to RV:
                <path to RV exe> <path to media 1> <start>-<end> <path to media 2> <start>-<end>

                Example:
                path/to/rv.exe path/to/media.####.exr 1001-1025 path/to/media.####.exr 1001-1040

        Args:
            media (:obj:`list` of :obj:`smartElements.media.Media`):
                Sequence of selected Media instances to use for command
                assembly.

        Returns:
            :obj:`list` of :obj:`str`: The command to run in order to send
                media to DJV.

        """
        command = [
            '"{}"'.format(self.get_executable())
        ]

        pattern = r"#+"
        for media_ in media:
            command.append('"{}"'.format(re.sub(pattern, "#", media_.path)))
            command.append("{}-{}".format(media_.first, media_.last))

        return command

    def label(self):
        """Return the label to use in the UI for this command.

        Returns:
            str: The label to use for the command in the UI.

        """
        return "Send to RV"

    def process(self, media):
        """Send selected media to RV.

        Args:
            media (:obj:`list` of :obj:`smartElements.media.Media`):
                Sequence of selected Media instances. Will send all selected
                media to RV.

        """
        command = " ".join(self.assemble_command(media))
        self.logger.info("Running %s using command: %s", self.name, command)

        subprocess.Popen(command, shell=True)
