"""Transcode a media using a given template."""

# Import built-in modules
import re
import subprocess

# Import local modules
from smartElements import paths
from smartElements.ingest_processors.base_ingest_processor import BaseIngestProcessor
from smartElements.constants import PATTERN_WRITE_SHORT_LINES


class Transcoder(BaseIngestProcessor):
    """Transcode a media using a given template."""

    def process(self):
        """Run the preview creation process."""
        command = self._assemble_command()
        command = " ".join(command)
        self.add_to_report("Transcoding using command: {0}".format(command))

        process = subprocess.Popen(
            command,
            stdout=subprocess.PIPE,
            stdin=subprocess.PIPE,
            stderr=subprocess.PIPE,
            universal_newlines=True,
            shell=True
        )

        # Poll process.stdout to show stdout live.
        while True:
            output = process.stdout.readline()
            if process.poll() is not None:
                break
            if output:
                output = output.strip()

                if re.match(PATTERN_WRITE_SHORT_LINES, output):
                    continue

                if output not in {"", "'b'"}:
                    self.add_to_report(output)

    def _assemble_command(self):
        """Assemble command to create preview in terminal mode.

        Returns:
            :obj:`list` of :obj:`str`: Command to run to create poster frames
                in terminal mode.

        """
        return [
            '"{0}"'.format(paths.get_nuke_exe().replace("\\", "/")),
            "-i",
            "-t",
            '"{0}"'.format(paths.get_terminal_script_transcode().replace("\\", "/")),
            '"{0}"'.format(paths.sanitize(self.job.media.path).replace("\\", "/")),
            '"{0}"'.format(paths.sanitize(self.job.parent_dst).replace("\\", "/")),
            '"template={}"'.format(paths.sanitize(self.settings["template"]))
        ]
