"""Create a preview for a media."""

# Import built-in modules
import os
import re
import subprocess

# Import local modules
from smartElements import paths
from smartElements.ingest_processors.base_ingest_processor import BaseIngestProcessor
from smartElements.constants import PATTERN_WRITE_SHORT_LINES


class PreviewCreator(BaseIngestProcessor):
    """Create a preview for a media."""

    @property
    def thumbnail_frame(self):
        """Get the validated configured thumbnail frame value.

        Notes:
            In case the user did not configure a valid value we will warn
            them about it and return the default value 'start'.

        Returns:
            str: The configured thumbnail frame value. Always returns one
                of ("start", "middle", "end", "auto").

        """
        value = self.settings["thumbnail_frame"]

        if value in ("start", "first"):
            return "start"

        elif value in ("middle", "midle"):
            return "middle"

        elif value in ("end", "last"):
            return "end"

        elif value == "auto":
            return "auto"

        else:
            self.logger.warning(
                "The thumbnail_frame value is set to an invalid value of "
                "'{}'. Choose from ['start', 'middle', 'end', 'auto']. "
                "Defaulting to 'start'.".format(self.settings["thumbnail_frame"])
            )
            return "start"

    def process(self):
        """Run the preview creation process."""
        dst = os.path.join(self.job.meta_dir, "seq")
        self.logger.debug("Creating sequence preview in: %s", dst)

        command = self._assemble_command(dst)
        self.add_to_report(
            "Creating sequence preview using: {}".format(" ".join(command))
        )

        process = subprocess.Popen(
            command,
            stdout=subprocess.PIPE,
            stdin=subprocess.PIPE,
            stderr=subprocess.PIPE,
            universal_newlines=True
        )

        # Poll process.stdout to show stdout live.
        while True:
            output = process.stdout.readline()
            if process.poll() is not None:
                break
            if output:
                output = output.strip()

                if re.match(PATTERN_WRITE_SHORT_LINES, output):
                    continue

                if output not in {"", "'b'"}:
                    self.add_to_report(output)

    def _assemble_command(self, dst):
        """Assemble command to create preview in terminal mode.

        Args:
            dst (str): Absolute path to render poster frames to.

        Returns:
            :obj:`list` of :obj:`str`: Command to run to create poster frames
                in terminal mode.

        """
        return [
            paths.get_nuke_exe(),
            "-i",
            "-t",
            paths.get_terminal_script_preview(),
            self.job.media.path,
            dst,
            "width={}".format(self.settings["width"]),
            "start={}".format(self.job.media.first),
            "end={}".format(self.job.media.last),
            "thumbnail_frame={}".format(self.thumbnail_frame)
        ]
