"""Extract information about a media."""

# Import built-in modules
import json
import os
import subprocess

# Import local modules
from smartElements import paths
from smartElements.media import Media
from smartElements.constants import ENV_NUKE_EXE
from smartElements.constants import PY2
from smartElements.constants import PY3
from smartElements.ingest_processors.base_ingest_processor import BaseIngestProcessor


class DataExtractor(BaseIngestProcessor):
    """Extract information about the media."""

    def process(self):
        """Run the data extraction process."""
        command = self._assemble_command()
        self.add_to_report("Extracting data using: {}".format(" ".join(
            command)))
        process = subprocess.Popen(
            command, stdout=subprocess.PIPE, stdin=subprocess.PIPE,
            stderr=subprocess.PIPE
        )

        data = {}
        if PY3:
            data = self.extract_data_py3(process)
        elif PY2:
            data = self.extract_data_py2(process)

        for line in process.stdout:
            line = str(line.rstrip())
            self.add_to_report(line)

        target_media = Media(self.get_destination_media_path())
        if not target_media:
            target_media = self.job.media

        if not data:
            data = {}

        data["name"] = target_media.base
        data["id"] = self.job.id
        data["size"] = target_media.get_size()
        data["source"] = self.job.media.path
        data["comment"] = ""
        data["type"] = os.path.splitext(target_media.path)[1]

        metadata_path = self._assemble_meta_path()
        self.logger.debug("Writing meta data to: %s", metadata_path)
        with open(metadata_path, "w") as file_:
            json.dump(data, file_, indent=4)

    def extract_data_py2(self, process):
        """Extract string passed dict from subprocess with python-2.

        Args:
            process (subprocess.Popen): The processor to analyse.

        Returns:
            dict the extracted string passed dict from subprocess.

        """
        for line in process.stdout:
            line = str(line.rstrip())
            self.add_to_report(line)
            if line.startswith("DATA="):
                line = line.replace("DATA=", "")
                line = line.split("ENDDATA")[0]

                return json.loads(line)

    def extract_data_py3(self, process):
        """Extract string passed dict from subprocess with python-3.

        Notes:
            Using the python-2 extraction the subprocess keeps hanging
            (see: https://bit.ly/3xbyWEm). We will need to break after byte
            lines. Attention, these are no byte types, but rather strings that
            'look' like a byte type by starting with -> b' <-, however, these
            are still a string.

        Args:
            process (subprocess.Popen): The processor to analyse.

        Returns:
            dict the extracted string passed dict from subprocess.

        """
        for line in iter(process.stdout.readline, b''):
            line = str(line.rstrip())
            self.add_to_report(line)

            if line.startswith("b'DATA="):
                line = line.replace("b'DATA=", "")
                line = line.split("ENDDATA")[0]

                return json.loads(line)

    def _assemble_command(self):
        """Assemble command to extract data in terminal mode.

        Returns:
            :obj:`list` of :obj:`str`: Command to run to extract data in
                terminal mode.

        """
        nuke_exe = os.environ.get(ENV_NUKE_EXE)
        if not nuke_exe:
            # Perform late import so that we only import a Nuke dependency
            # when absolutely necessary.
            from smartElements import paths
            nuke_exe = paths.get_nuke_exe()

        return [
            nuke_exe,
            "-i",
            "-t",
            paths.get_terminal_script_extract_data(),
            self.get_destination_media_path(),
            self.job.parent_dst
        ]

    def get_destination_media_path(self):
        """Get the media in the destination root.

        Returns:
            str: Absolute path of first file in the destination root.

        """
        files = [
            os.path.join(self.job.parent_dst, name)
            for name in os.listdir(self.job.parent_dst)
            if not name.startswith(".")
            and os.path.isfile(os.path.join(self.job.parent_dst, name))
        ]

        # Files exist, i.e. hard copy to destination.
        if files:
            return files[0]
        # Files don't exist. i.e. soft copy using the original media location.
        else:
            return self.media.files[0]

    def _assemble_meta_path(self):
        """Assemble path to meta data file.

        Returns:
            str: Absolute path to meta data file.

        """
        return os.path.join(
            self.job.meta_dir, "meta.json"
        )
